/* wave.c */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * Bricks!                                  Copyright (C) 2001, AnTiKoNs *
 *---------/                                                             *
 * This program is free software; you can redistribute it and/or modify  *
 * it under the terms of the GNU General Public License as published by  *
 * the Free Software Foundation; either version 2 of the License, or     *
 * (at your option) any later version.                                   *
 *                                                                       *
 * This program is distributed in the hope that it will be useful,       *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 * GNU General Public License for more details.                          *
 *                                                                       *
 * The author of this program may be contacted at antikons@ifrance.com   *
 *                                                                       *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

// Un fichier assez complexe pour charger des .WAV  partir d'un fichier ou d'une ressource

#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>
#include "_error.h"
#include "wave.h"
#include "resource.h"


char* g_pData = NULL;		// pointeur vers le dbut des donnes charges
char* g_pCurrent = NULL;	// pointeur vers o on en est des donnes charges


// Fonction appele par Windows pour charger un .WAV  partir d'une ressource (=un fichier inclu dans l'excutable)
LRESULT CALLBACK MMIOProc(
    LPSTR lpmmioinfo,	
    UINT uMsg,	
    LONG lParam1,	
    LONG lParam2 	
   )
{
	int res = 0;
	HRSRC hResInfo = NULL;
	HGLOBAL hResData = NULL;
	HINSTANCE h = GetModuleHandle(NULL);
	
	switch(uMsg)
	{
	
	// Ouverture de la ressource
	case MMIOM_OPEN:
		
		g_pCurrent = g_pData = NULL;
		
		res = atoi( (const char*)lParam1 );
		hResInfo = FindResource( h, MAKEINTRESOURCE(res), "WAVE" );
		if( hResInfo == NULL )
		{
			Error("MMIOProc : FindResource (%i)", GetLastError() );
			return MMIOERR_CANNOTOPEN;
		}
		
		hResData = LoadResource( h, hResInfo );
		if( hResData == NULL )
		{
			Error("MMIOProc : LoadResource (%i)", GetLastError() );
			return MMIOERR_CANNOTOPEN;
		}
		
		g_pCurrent = g_pData = (char*)LockResource( hResData );
		if( g_pData == NULL )
		{
			Error("MMIOProc : LockResource" );
			return MMIOERR_OUTOFMEMORY;
		}
		return MMSYSERR_NOERROR;
	
	// Fermeture
	case MMIOM_CLOSE:
		return 0;
	
	// Lit des donnes
	case MMIOM_READ:
		memcpy( (void*)lParam1, g_pCurrent, lParam2 );
		g_pCurrent += lParam2;
		return lParam2;
	
	// Change la position dans le fichier
	case MMIOM_SEEK:
		switch(lParam2)
		{
		case SEEK_CUR:
			g_pCurrent += lParam1;
			return lParam1;
		case SEEK_END:
			return -1;
		case SEEK_SET:
			g_pCurrent = g_pData + lParam1;
			return lParam1;
		}
		return -1;
	
	// Ecrit des donnes
	case MMIOM_WRITE:
	case MMIOM_WRITEFLUSH:
	default:
		Error("MMIOProc : trying to write");
		return -1;
	}
	Error("MMIOProc : Unknown message");
	return 0;
}


//Same as mmioOpen
HMMIO mmioLoadResource( LPSTR lpName, LPMMIOINFO lpmmioinfo, DWORD dwOpenFlags )
{
	static char buf[64];
	MMIOINFO mmioinfo;
	HMMIO hMmio;
	LPMMIOPROC pIOProc;
	FOURCC fcc;
	
	if( lpmmioinfo == NULL )
	{
		lpmmioinfo = &mmioinfo;
		ZeroMemory( lpmmioinfo, sizeof(MMIOINFO) );
	}
	
	fcc = mmioStringToFOURCC("RESO", MMIO_TOUPPER);
	
	pIOProc = mmioInstallIOProc( fcc, MMIOProc, MMIO_INSTALLPROC );
	if( pIOProc == NULL )
	{
		Error("mmioLoadResource : mmioInstallIOProc");
		return NULL;
	}
	
	lpmmioinfo->pIOProc = pIOProc;
	
	wsprintf( buf, "%i", lpName );
	
	hMmio = mmioOpen( buf, lpmmioinfo, dwOpenFlags );
	if( lpmmioinfo->wErrorRet == MMIOERR_PATHNOTFOUND )
		OutputDebugString("MMIOERR_PATHNOTFOUND\n");
	if( hMmio == NULL )
	{
		Error("mmioLoadResource : mmioOpen(%i)", lpmmioinfo->wErrorRet);
	}
	
	mmioInstallIOProc( fcc, MMIOProc, MMIO_REMOVEPROC );
	
	return hMmio;
}


/* This function will open a wave input file and prepare it for reading,
 * so the data can be easily
 * read with WaveReadFile. Returns 0 if successful, the error code if not.
 *      pszFileName - Input filename to load.
 *      phmmioIn    - Pointer to handle which will be used
 *          for further mmio routines.
 *      ppwfxInfo   - Ptr to ptr to WaveFormatEx structure
 *          with all info about the file.                        
 *      
*/
int WaveOpenFile(
    TCHAR*pszFileName,                              // (IN)
    HMMIO *phmmioIn,                                // (OUT)
    WAVEFORMATEX **ppwfxInfo,                       // (OUT)
    MMCKINFO *pckInRIFF                             // (OUT)
            )
{
    HMMIO           hmmioIn;
    MMCKINFO        ckIn;           // chunk info. for general use.
    PCMWAVEFORMAT   pcmWaveFormat;  // Temp PCM structure to load in.       
    WORD            cbExtraAlloc;   // Extra bytes for waveformatex 
    int             nError;         // Return value.


    // Initialization...
    *ppwfxInfo = NULL;
    nError = 0;
    hmmioIn = NULL;
    
	if ((hmmioIn = mmioLoadResource(pszFileName, NULL, MMIO_ALLOCBUF | MMIO_READ)) == NULL)
	{
		if ((hmmioIn = mmioOpen(pszFileName, NULL, MMIO_ALLOCBUF | MMIO_READ)) == NULL)
		{
			nError = ER_CANNOTOPEN;
			goto ERROR_READING_WAVE;
		}
	}
	
	
    if ((nError = (int)mmioDescend(hmmioIn, pckInRIFF, NULL, 0)) != 0)
        {
        goto ERROR_READING_WAVE;
        }


    if ((pckInRIFF->ckid != FOURCC_RIFF) || (pckInRIFF->fccType != mmioFOURCC('W', 'A', 'V', 'E')))
        {
        nError = ER_NOTWAVEFILE;
        goto ERROR_READING_WAVE;
        }
            
    /* Search the input file for for the 'fmt ' chunk.     */
    ckIn.ckid = mmioFOURCC('f', 'm', 't', ' ');
    if ((nError = (int)mmioDescend(hmmioIn, &ckIn, pckInRIFF, MMIO_FINDCHUNK)) != 0)
        {
        goto ERROR_READING_WAVE;                
        }
                    
    /* Expect the 'fmt' chunk to be at least as large as <PCMWAVEFORMAT>;
    * if there are extra parameters at the end, we'll ignore them */
    
    if (ckIn.cksize < (long) sizeof(PCMWAVEFORMAT))
        {
        nError = ER_NOTWAVEFILE;
        goto ERROR_READING_WAVE;
        }
                                                            
    /* Read the 'fmt ' chunk into <pcmWaveFormat>.*/     
    if (mmioRead(hmmioIn, (HPSTR) &pcmWaveFormat, (long) sizeof(pcmWaveFormat)) != (long) sizeof(pcmWaveFormat))
        {
        nError = ER_CANNOTREAD;
        goto ERROR_READING_WAVE;
        }
                            

    // Ok, allocate the waveformatex, but if its not pcm
    // format, read the next word, and thats how many extra
    // bytes to allocate.
    if (pcmWaveFormat.wf.wFormatTag == WAVE_FORMAT_PCM)
        cbExtraAlloc = 0;                               
                            
    else
        {
        // Read in length of extra bytes.
        if (mmioRead(hmmioIn, (LPTSTR) &cbExtraAlloc,
            (long) sizeof(cbExtraAlloc)) != (long) sizeof(cbExtraAlloc))
            {
            nError = ER_CANNOTREAD;
            goto ERROR_READING_WAVE;
            }

        }
                            
    // Ok, now allocate that waveformatex structure.
    if ((*ppwfxInfo = (WAVEFORMATEX*)GlobalAlloc(GMEM_FIXED, sizeof(WAVEFORMATEX)+cbExtraAlloc)) == NULL)
        {
        nError = ER_MEM;
        goto ERROR_READING_WAVE;
        }

    // Copy the bytes from the pcm structure to the waveformatex structure
    memcpy(*ppwfxInfo, &pcmWaveFormat, sizeof(pcmWaveFormat));
    (*ppwfxInfo)->cbSize = cbExtraAlloc;

    // Now, read those extra bytes into the structure, if cbExtraAlloc != 0.
    if (cbExtraAlloc != 0)
        {
        if (mmioRead(hmmioIn, (LPTSTR) (((BYTE*)&((*ppwfxInfo)->cbSize))+sizeof(cbExtraAlloc)),
            (long) (cbExtraAlloc)) != (long) (cbExtraAlloc))
            {
            nError = ER_NOTWAVEFILE;
            goto ERROR_READING_WAVE;
            }
        }

    /* Ascend the input file out of the 'fmt ' chunk. */                                                            
    if ((nError = mmioAscend(hmmioIn, &ckIn, 0)) != 0)
        {
        goto ERROR_READING_WAVE;

        }
    

    goto TEMPCLEANUP;               

ERROR_READING_WAVE:
    if (*ppwfxInfo != NULL)
        {
        GlobalFree(*ppwfxInfo);
        *ppwfxInfo = NULL;
        }               

    if (hmmioIn != NULL)
    {
    mmioClose(hmmioIn, 0);
        hmmioIn = NULL;
        }
    
TEMPCLEANUP:
    *phmmioIn = hmmioIn;

    return(nError);

}

/*      This routine has to be called before WaveReadFile as it searchs for the chunk to descend into for
    reading, that is, the 'data' chunk.  For simplicity, this used to be in the open routine, but was
    taken out and moved to a separate routine so there was more control on the chunks that are before
    the data chunk, such as 'fact', etc... */
int WaveStartDataRead(
                    HMMIO *phmmioIn,
                    MMCKINFO *pckIn,
                    MMCKINFO *pckInRIFF
                    )
{
    int                     nError;

    nError = 0;
    
    // Do a nice little seek...
	if ((nError = mmioSeek(*phmmioIn, pckInRIFF->dwDataOffset + sizeof(FOURCC), SEEK_SET)) == -1)
	{
		Error("WaveStartDataRead : mmioSeek");
		goto CLEANUP;
	}
	
	nError = 0;
	//      Search the input file for for the 'data' chunk.
	pckIn->ckid = mmioFOURCC('d', 'a', 't', 'a');
	if ((nError = mmioDescend(*phmmioIn, pckIn, pckInRIFF, MMIO_FINDCHUNK)) != 0)
	{
		goto ERROR_READING_WAVE;
	}
	
	goto CLEANUP;
	
ERROR_READING_WAVE:
	
CLEANUP:        
	return(nError);
}


/*      This will read wave data from the wave file.  
        Make sure we're descended into
        the data chunk, else this will fail bigtime!

    hmmioIn         - Handle to mmio.
    cbRead          - # of bytes to read.   
    pbDest          - Destination buffer to put bytes.
    cbActualRead- # of bytes actually read.
*/
int WaveReadFile(
        HMMIO hmmioIn,                          // IN
        UINT cbRead,                            // IN           
        BYTE *pbDest,                           // IN
        MMCKINFO *pckIn,                        // IN
        UINT *cbActualRead                      // OUT        
        )
{

    MMIOINFO    mmioinfoIn;         // current status of <hmmioIn>
    int                     nError;
    UINT            cT, cbDataIn;

    nError = 0;

    if (nError = mmioGetInfo(hmmioIn, &mmioinfoIn, 0) != 0)
        {
        goto ERROR_CANNOT_READ;
        }
                
    cbDataIn = cbRead;
    if (cbDataIn > pckIn->cksize) 
        cbDataIn = pckIn->cksize;       

    pckIn->cksize -= cbDataIn;
    
    for (cT = 0; cT < cbDataIn; cT++)
        {
        /* Copy the bytes from the io to the buffer. */
        if (mmioinfoIn.pchNext == mmioinfoIn.pchEndRead)
            {
        if ((nError = mmioAdvance(hmmioIn, &mmioinfoIn, MMIO_READ)) != 0)
                {
        goto ERROR_CANNOT_READ;
                } 
        if (mmioinfoIn.pchNext == mmioinfoIn.pchEndRead)
                {
                nError = ER_CORRUPTWAVEFILE;
        goto ERROR_CANNOT_READ;
                }
            }
            
        // Actual copy.
		mmioinfoIn.pchNext++;
		*((BYTE*)pbDest+cT) = *((BYTE*)mmioinfoIn.pchNext);
	}

    if ((nError = mmioSetInfo(hmmioIn, &mmioinfoIn, 0)) != 0)
        {
        goto ERROR_CANNOT_READ;
        }

    *cbActualRead = cbDataIn;
    goto FINISHED_READING;

ERROR_CANNOT_READ:
    *cbActualRead = 0;

FINISHED_READING:
    return(nError);

}

/*      This will close the wave file opened with WaveOpenFile.  
    phmmioIn - Pointer to the handle to input MMIO.
    ppwfxSrc - Pointer to pointer to WaveFormatEx structure.

    Returns 0 if successful, non-zero if there was a warning.

*/
int WaveCloseReadFile(
            HMMIO *phmmio,                          // IN
            WAVEFORMATEX **ppwfxSrc                 // IN
            )
{

    if (*ppwfxSrc != NULL)
        {
        GlobalFree(*ppwfxSrc);
        *ppwfxSrc = NULL;
        }

    if (*phmmio != NULL)
        {
        mmioClose(*phmmio, 0);
        *phmmio = NULL;
        }

    return(0);

}


/*      This routine loads a full wave file into memory.  
        Be careful, wave files can get
        pretty big these days :).  

    szFileName      -       sz Filename
    cbSize          -       Size of loaded wave (returned)
    cSamples        -       # of samples loaded.
    ppwfxInfo       -       Pointer to pointer to waveformatex structure.  
                            The wfx structure
                            IS ALLOCATED by this routine!  Make sure to free it!
    ppbData         -       Pointer to a byte pointer (globalalloc) which is allocated by this 
                    routine.  Make sure to free it!

    Returns 0 if successful, else the error code.
*/
int WaveLoadFile(
            TCHAR*pszFileName,                      // (IN)
            UINT *cbSize,                           // (OUT)
            WAVEFORMATEX **ppwfxInfo,				// (OUT)
            BYTE **ppbData                          // (OUT)
            )
{

    HMMIO                           hmmioIn;        
    MMCKINFO                        ckInRiff;
    MMCKINFO                        ckIn;
    int                                     nError;
    UINT                            cbActualRead;

    *ppbData = NULL;
    *ppwfxInfo = NULL;
    *cbSize = 0;
    
    if ((nError = WaveOpenFile(pszFileName, &hmmioIn, ppwfxInfo, &ckInRiff)) != 0)
        {
        goto ERROR_LOADING;
        }

    if ((nError = WaveStartDataRead(&hmmioIn, &ckIn, &ckInRiff)) != 0)
        {
        goto ERROR_LOADING;
        }

    // Ok, size of wave data is in ckIn, allocate that buffer.
    if ((*ppbData = (BYTE *)GlobalAlloc(GMEM_FIXED, ckIn.cksize)) == NULL)
        {
        nError = ER_MEM;
        goto ERROR_LOADING;
        }

    if ((nError = WaveReadFile(hmmioIn, ckIn.cksize, *ppbData, &ckIn, &cbActualRead)) != 0)
        {
        goto ERROR_LOADING;
        }        
    
    *cbSize = cbActualRead;
    goto DONE_LOADING;

ERROR_LOADING:
    if (*ppbData != NULL)
        {
        GlobalFree(*ppbData);
        *ppbData = NULL;
        }
    if (*ppwfxInfo != NULL)
        {
        GlobalFree(*ppwfxInfo);
        *ppwfxInfo = NULL;
        }
            
DONE_LOADING:
    // Close the wave file. 
    if (hmmioIn != NULL)
        {
        mmioClose(hmmioIn, 0);
        hmmioIn = NULL;
        }

    return(nError);
}
